/**
 * This class implements the methods of the ControllerInterface.
 * Therefore it creates a new MyActionListener which will be given to the GUI.
 * There is also the possibility to create a new instance of itself and provides it for the GUI. 
 * 
 * @author Andreas Arnold, Fabian Kueng, Patric Uebersax
 * @version Version 0.10
 */
package control;

import gui.GUI;
import gui.GUIInterface;
import gui.SearchField;

import java.util.Collection;
import java.util.Date;

import control.search.AuthorSearch;
import control.search.FromDateSearch;
import control.search.FullTextSearch;
import control.search.HierachySearch;
import control.search.SearchCriteria;
import control.search.TitleSearch;
import control.search.ToDateSearch;

import db.DBInterface;
import db.Singelton;
import db.info.Info;

public class ControllerImplementation implements ControllerInterface{
	
	/**
	 * A reference for itself.
	 */
	private static ControllerImplementation ref;
	private GUIInterface gui;
	private DBInterface database;
	
	/**
	 * Creates and returns a reference to the current instance of the ControllerImplementation. It will just create a new reference, when there isn't a reference yet.
	 * @return Returns a reference for the ControllerImplementation
	 */
	
	public ControllerImplementation() {
		gui = GUI.getInstance();
		database = Singelton.getInstance();
	}
	public static ControllerInterface getInstance() {
		if(ref == null) {
			ref = new ControllerImplementation();
		}
		return ref;
	}

	/**
	 * Gets all SearchFields from the GUI and makes a Collection with all the search
	 * words to check. If no word was entered in the GUI, the corresponding search won't
	 * start.
	 * @param searchWord All SearchFields from the GUI-Frame
	 * @return The collection with all results.
	 */
	public Collection search(SearchField searchWord) {
		SearchCriteria sc = new SearchCriteria();
		
		// Check for entered searchwords.
		String titleSearch = searchWord.getTitle();
		String fullTextSearch = searchWord.getFullText();
		Collection hierarchySearch = searchWord.getHierarchy();
		Date fromDateSearch = searchWord.getFromDate();
		Date toDateSearch = searchWord.getToDate();
		String authorSearch = searchWord.getAuthor();
		
		// Generate for every entered word the corresponding search.
		if(!(titleSearch.equals(""))) {
			sc.add(new TitleSearch(titleSearch));
		}
		
		if(!(fullTextSearch.equals(""))) {
			sc.add(new FullTextSearch(fullTextSearch));
		}
		
		if(null != hierarchySearch && hierarchySearch.size() > 0) {
			sc.add(new HierachySearch(hierarchySearch));
		}
		
		if(null != fromDateSearch) {
			sc.add(new FromDateSearch(fromDateSearch));
		}
		
		if(null != toDateSearch) {
			sc.add(new ToDateSearch(toDateSearch));
		}
		
		if(!(authorSearch.equals(""))) {
			sc.add(new AuthorSearch(authorSearch));
		}
		
		// Deliever the SearchCriterias to the database and get the results.
		Collection c = database.getResultSet(sc);
		
		return c;
	}

	public void remove(Info info) {
		database.removeInfo(info);
	}

	public void save(Info info) {
		database.saveInfo(info);
	}
}
